package easik.sketch.util.Export.Constraints;

import java.util.HashMap;
import java.util.LinkedList;

import easik.sketch.util.Export.ExportConstants;
import easik.sketch.util.Export.Components.ExportEdge;
import easik.sketch.util.Export.Components.ExportPath;

/**
 * A class to create a sum constraint object, including all necessary information gleaned 
 * from the XML file.  Should not be used for any other purpose than database exportation, as it 
 * contains no graphical information.
 * 
 * @author Vera Ranieri 2006
 * @author Kevin Green 2006
 * @since 2006-05-22 Vera Ranieri
 * @version 2006-08-28 Kevin Green
 *
 */
public class ExportSumConstraint extends ExportConstraint {

	/**
	 * The delete procedure name
	 */
	private final static String DEL_PROC_NAME = "sumDel";
	/**
	 * The insert procedure name
	 */
	private final static String INS_PROC_NAME = "sumIns";
	//private final static String DEL = ;
	//private final static String USING = ;
	//private final static String WHERE = ;
	/**
	 * A hashmap of the target tables of the first edge in the paths leading from the pullback domain, 
	 * indexed by the name of the domain of the path.
	 */
	private HashMap<String, String> _varTable;
	
	/**
	 * Constructor.
	 * @param id A unique identifier for this constraint.
	 */
	public ExportSumConstraint(int id){
		super();
		_id = id;
		_varTable = new HashMap<String, String>();
	}
	
	/**
	 * Sets the base table in the superclass based on requirements for a sum constraint.  This base table
	 * is the codomain of all paths included in this constraint.
	 */
	public void setBaseTable(){
		ExportPath basePath = (ExportPath)_paths.get(0);
		_baseTable = basePath.getCoDomain();
	}
	
	/**
	 * Sets all constraint strings for this constraint.
	 * 
	 * @since 2006-06-22 Vera Ranieri
	 */
	public void setConstraintStrings(){
		setDeleteProcs();
		setInsertProcs();
	}
	
	/**
	 * Sets the delete procedure strings for this constraint, and returns it.  One delete procedure string is
	 * created for each table contributing to the sum constraint.
	 * .
	 * @since 2006-06-09 Vera Ranieri
	 */
	private void setDeleteProcs(){
		
		String currentString;
		String currentName;
		
		for(ExportPath currentPath: _paths){
			
			currentName = _db_name + "_" + DEL_PROC_NAME +(String)currentPath.getDomain()+_id;
			_b_d_tables.put(currentPath.getDomain(),currentName);
			
			currentString = ExportConstants.CREATE_PROC + currentName + ExportConstants.PROC_PARAM;
			currentString += ExportConstants.BEGIN;
			currentString += "DELETE FROM "+currentPath.getCoDomain() + " "+ "USING " + ExportPath.getAllTablesString(currentPath) 
							+ "WHERE \n"+ExportPath.getDelConditions(currentPath) +"; \n";
			currentString += ExportConstants.END;
			_procedureStrings.add(currentString);
		}
		
	}
	
	/**
	 * 
	 * Sets the insert procedure strings for this constraint, and returns it.  One insert procedure string is
	 * created for each table contributing to the sum constraint.
	 * <br>
	 * When called, the procedure adds an element along the path, beginning at the base table (i.e. the sum table)
	 * to ensure an insert is allowed.  By using the SQL function <b>LAST_INSERT_ID()</b>, it is ensured
	 * that each entry is new and unique.
	 * 
	 * @since 2006-06-21 Vera Ranieri
	 */
	private void setInsertProcs(){
		
		String currentString, currentPathDomain;
		String currentName;
		
		for(ExportPath currentPath: _paths){
				
				currentPathDomain = currentPath.getDomain();
				currentName = _db_name + "_" + INS_PROC_NAME +currentPathDomain+_id;
				
				_b_i_tables.put(currentPathDomain,currentName);
				_varTable.put(currentPathDomain, currentPath.getEdges().getFirst().getTarget());
	
				currentString = ExportConstants.CREATE_PROC + currentName + ExportConstants.PROC_PARAM;
				currentString += ExportConstants.BEGIN;
				
				currentString += getInsertString(currentPath);
				currentString += ExportConstants.END;
				_procedureStrings.add(currentString);
			
		}
		
	}
	
	/**
	 * Formats the<b> INSERT</b> string contained inside the sum procedure
	 * @param path The path for which this procedure is being called
	 * @return The string of the insert, formatted for SQL
	 * @since 2006-06-21 Vera Ranieri
	 */
	private String getInsertString(ExportPath path){
		
		LinkedList<ExportEdge> edges = path.getEdges();
		if(edges.size()==1){
			return "INSERT INTO " +path.getCoDomain() +" VALUES (id); \n";
		}
		else{
			String stmt = "INSERT INTO " +path.getCoDomain() + " VALUES (); \n";

			String source, target;
			for(int i =edges.size()-1; i>1 ; i --){
				source = edges.get(i).getSource();
				target = edges.get(i).getTarget();
				stmt += "INSERT INTO " + source + "(" + target + ExportConstants.ID + ") VALUES (LAST_INSERT_ID()); \n";
				
			}
			ExportEdge e = edges.get(1);
			stmt += "INSERT INTO " + e.getSource() +"("+ e.getSource() + ExportConstants.ID +", " 
					+ e.getTarget() + ExportConstants.ID +") VALUES (id, LAST_INSERT_ID()); \n";
			
			return stmt;
		}
	}
	
	/**
	 * Gets the target table of the first edge in the path that has as its domain, the table <it>inTable</it>.  This
	 * is required so that triggers can pass the correct parameter to the <b>BEFORE INSERT</b> procedure.
	 * 
	 * @param inTable The table which is the domain of the path for which the trigger is being called.
	 * @return The string representation of the codomain of the first edge of the path.
	 * @since 2006-07-13 Vera Ranieri
	 */
	public String getVarTable(String inTable){
		return _varTable.get(inTable);
	}
	
}
